using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Drawing.Text;
using System.Windows.Forms;


namespace WebTracker
{
	public class ShapeCollection : ReadOnlyCollectionBase
	{
		Canvas canvas;
		internal ShapeCollection(Canvas canvas)
		{
			this.canvas = canvas;
		}
		public object SyncRoot
		{
			get
			{
				return this.InnerList.SyncRoot;
			}
		}
		public Shape Add(Shape a)
		{
			if(null != a.canvas)
			{
				throw new InvalidOperationException("Shape already added to another canvas.");
			}
			a.canvas = this.canvas;
			this.InnerList.Insert(0, a);
			this.canvas.Invalidate(a);
			this.canvas.NotifyShapeAdded(a.eventArgs);
			return a;
		}
		internal void BringToFront(Shape a)
		{
			this.InnerList.Remove(a);
			this.InnerList.Insert(0, a);
		}
		internal void SendToBack(Shape a)
		{
			this.InnerList.Remove(a);
			this.InnerList.Add(a);
		}
		internal void BringForward(Shape a)
		{
			int index = this.InnerList.IndexOf(a);
			this.InnerList.RemoveAt(index);
			this.InnerList.Insert(Math.Max(0, index - 1), a);
		}
		internal void SendBackward(Shape a)
		{
			int index = this.InnerList.IndexOf(a);
			this.InnerList.RemoveAt(index);
			index++;
			if(index >= this.InnerList.Count)
			{
				this.InnerList.Add(a);
			}
			else
			{
				this.InnerList.Insert(index, a);
			}
		}
		internal int IndexOf(Shape a)
		{
			return this.InnerList.IndexOf(a);
		}
		public void Remove(Shape a)
		{
			if(this.InnerList.Contains(a))
			{
				a.canvas = null;
				this.InnerList.Remove(a);
				this.canvas.Invalidate(a);
				this.canvas.NotifyShapeRemoved(a.eventArgs);
			}
		}
		public Shape this[int index]
		{
			get
			{
				return this.InnerList[index] as Shape;
			}
		}
		public void Clear()
		{
			ArrayList al = this.InnerList.Clone() as ArrayList;
			this.InnerList.Clear();
			this.canvas.Refresh();
			foreach(Shape a in al)
			{
				a.canvas = null;
				this.canvas.NotifyShapeRemoved(new ShapeEventArgs(a));
			}
		}
	}

	public enum HandleVisibility
	{
		Never,
		Always,
		WhenSelected
	}
	public class ShapeEventArgs : EventArgs
	{
		private Shape shape;
		private bool editable;
		public Shape Shape
		{
			get
			{
				return this.shape;
			}
			set
			{
				if(!this.editable)
				{
					throw new InvalidOperationException("Property is read-only.");
				}
				this.shape = value;
			}
		}
		public bool Editable
		{
			get
			{
				return this.editable;
			}
		}
		public ShapeEventArgs()
		{
			this.editable = true;
			this.shape = null;
		}
		public ShapeEventArgs(Shape shape)
		{
			this.editable = false;
			this.shape = shape;
		}
	}
	public delegate void ShapeEventHandler(object sender, ShapeEventArgs e);
	public enum HitTestCode
	{
		None = 0,
		SizeNW = 1,
		SizeN = 2,
		SizeNE = 3,
		SizeE = 4,
		SizeSE = 5,
		SizeS = 6,
		SizeSW = 7,
		SizeW = 8,
		Move = 9
	}
	public class Shape
	{
		private static Font DefaultFont = new Font("Tahoma", 8);
		private static StringFormat DefaultStringFormat = new StringFormat(StringFormat.GenericDefault);
		internal ShapeEventArgs eventArgs;
		internal Canvas canvas = null;
		private void Refresh()
		{
			if(null != this.canvas)
			{
				this.canvas.BeginUpdate();
				this.canvas.Invalidate(this);
				this.canvas.EndUpdate();
			}
		}
		private Rectangle rectangle = Rectangle.Empty;
		public Rectangle Rectangle
		{
			get
			{
				return this.rectangle;
			}
			set
			{
				if(value != this.rectangle)
				{
					if(null != this.canvas)
					{
						this.canvas.BeginUpdate();
						this.canvas.Invalidate(this);
					}
					this.rectangle = new Rectangle(Math.Max(0, value.Left), Math.Max(0, value.Top), value.Width, value.Height);
					if(null != this.canvas)
					{
						this.canvas.Invalidate(this.Bounds);
						this.canvas.EndUpdate();
						this.canvas.NotifyShapeBoundsChanged(this.eventArgs);
					}
				}
			}
		}
		private string text = string.Empty;
		public string Text
		{
			get
			{
				return this.text;
			}
			set
			{
				this.text = value;
				this.Refresh();
			}
		}
		private Color fillColor = Color.Transparent;
		public Color FillColor
		{
			get
			{
				return this.fillColor;
			}
			set
			{
				this.fillColor = value;
				this.Refresh();
			}
		}
		private Image image = null;
		public Image Image
		{
			get
			{
				return this.image;
			}
			set
			{
				this.image = value;
				this.Refresh();
			}
		}
		private Rectangle imageRectangle = Rectangle.Empty;
		public Rectangle ImageRectangle
		{
			get
			{
				if(this.imageRectangle == Rectangle.Empty)
				{
					if(null == this.image)
					{
						return Rectangle.Empty;
					}
					else
					{
						return new Rectangle(new Point(0, 0), this.image.Size);
					}
				}
				else
				{
					return this.imageRectangle;
				}
			}
			set
			{
				this.imageRectangle = value;
				this.Refresh();
			}
		}
		private Color foreColor = Color.Black;
		public Color ForeColor
		{
			get
			{
				return this.foreColor;
			}
			set
			{
				this.foreColor = value;
				this.Refresh();
			}
		}
		private Color backColor = Color.White;
		public Color BackColor
		{
			get
			{
				return this.backColor;
			}
			set
			{
				this.backColor = value;
				this.Refresh();
			}
		}
		private Font font = DefaultFont;
		public Font Font
		{
			get
			{
				return this.font;
			}
			set
			{
				if(null == value)
				{
					this.font = DefaultFont;
				}
				else
				{
					this.font = value;
				}
			}
		}
		private bool visible = true;
		public bool Visible
		{
			get
			{
				return this.visible;
			}
			set
			{
				this.visible = value;
				this.Refresh();
			}
		}
		private bool selected = false;
		public bool Selected
		{
			get
			{
				return this.selected;
			}
			set
			{
				if(value != this.selected)
				{
					this.selected = value;
					this.Refresh();
					if(null != this.canvas)
					{
						this.canvas.NotifySelectionChanged(this.eventArgs);
					}
				}
			}
		}
		private bool enabled = true;
		public bool Enabled
		{
			get
			{
				return this.enabled;
			}
			set
			{
				this.enabled = value;
				this.Refresh();
			}
		}
		private object tag = null;
		public object Tag
		{
			get
			{
				return this.tag;
			}
			set
			{
				this.tag = value;
			}
		}
		public int ZIndex
		{
			get
			{
				if(null == this.canvas)
				{
					return 0;
				}
				else
				{
					return this.canvas.Shapes.IndexOf(this);
				}
			}
		}
		private Size handleSize = new Size(9, 9);
		public Size HandleSize
		{
			get
			{
				return this.handleSize;
			}
			set
			{
				this.handleSize = value;
				this.Refresh();
				if(null != this.canvas)
				{
					this.canvas.NotifyShapeBoundsChanged(this.eventArgs);
				}
			}
		}
		private int borderThickness = 5;
		public int BorderThickness
		{
			get
			{
				return this.borderThickness;
			}
			set
			{
				this.borderThickness = value;
				this.Refresh();
				if(null != this.canvas)
				{
					this.canvas.NotifyShapeBoundsChanged(this.eventArgs);
				}
			}
		}
		private HatchStyle borderStyle = HatchStyle.DarkDownwardDiagonal;
		public HatchStyle BorderStyle
		{
			get
			{
				return this.borderStyle;
			}
			set
			{
				this.borderStyle = value;
				this.Refresh();
			}
		}
		private bool textVisible = true;
		public bool TextVisible
		{
			get
			{
				return this.textVisible;
			}
			set
			{
				this.textVisible = value;
				this.Refresh();
			}
		}
		private bool textLocked = false;
		public bool TextLocked
		{
			get
			{
				return this.textLocked;
			}
			set
			{
				this.textLocked = value;
			}
		}
		private ContentAlignment textPlacement = ContentAlignment.TopCenter;
		public ContentAlignment TextPlacement
		{
			get
			{
				return this.textPlacement;
			}
			set
			{
				this.textPlacement = value;
				this.Refresh();
			}
		}
		private bool movable = true;
		public bool Movable
		{
			get
			{
				return this.movable;
			}
			set
			{
				this.movable = value;
			}
		}
		private bool resizable = true;
		public bool Resizable
		{
			get
			{
				return this.resizable;
			}
			set
			{
				this.resizable = value;
				this.Refresh();
				if(null != this.canvas)
				{
					this.canvas.NotifyShapeBoundsChanged(this.eventArgs);
				}
			}
		}
		private HandleVisibility handleVisibility = HandleVisibility.WhenSelected;
		public HandleVisibility HandleVisibility
		{
			get
			{
				return this.handleVisibility;
			}
			set
			{
				this.handleVisibility = value;
				this.Refresh();
				if(null != this.canvas)
				{
					this.canvas.NotifyShapeBoundsChanged(this.eventArgs);
				}
			}
		}
		private bool borderSelect = true;
		public bool BorderSelect
		{
			get
			{
				return this.borderSelect;
			}
			set
			{
				this.borderSelect = value;
			}
		}
		private Size minimumSize = new Size(12, 12);
		public Size MinimumSize
		{
			get
			{
				return this.minimumSize;
			}
			set
			{
				this.minimumSize = value;
			}
		}
		private Size maximumSize = new Size(int.MaxValue, int.MaxValue);
		public Size MaximumSize
		{
			get
			{
				return this.maximumSize;
			}
			set
			{
				this.maximumSize = value;
			}
		}
		public void BringToFront()
		{
			if(null != this.canvas)
			{
				this.canvas.Shapes.BringToFront(this);
				this.canvas.Refresh();
				this.canvas.NotifyShapeBoundsChanged(this.eventArgs);
			}
		}
		public void SendToBack()
		{
			if(null != this.canvas)
			{
				this.canvas.Shapes.SendToBack(this);
				this.canvas.Refresh();
				this.canvas.NotifyShapeBoundsChanged(this.eventArgs);
			}
		}
		public void BringForward()
		{
			if(null != this.canvas)
			{
				this.canvas.Shapes.BringForward(this);
				this.canvas.Refresh();
				this.canvas.NotifyShapeBoundsChanged(this.eventArgs);
			}
		}
		public void SendBackward()
		{
			if(null != this.canvas)
			{
				this.canvas.Shapes.SendBackward(this);
				this.canvas.Refresh();
				this.canvas.NotifyShapeBoundsChanged(this.eventArgs);
			}
		}
		public Shape()
		{
			eventArgs = new ShapeEventArgs(this);
		}
		public Shape(Canvas canvas) : this()
		{
			this.BackColor = canvas.BackColor;
			this.BorderSelect = canvas.BorderSelect;
			this.BorderStyle = canvas.BorderStyle;
			this.BorderThickness = canvas.BorderThickness;
			this.Font = canvas.Font;
			this.ForeColor = canvas.ForeColor;
			this.HandleSize = canvas.HandleSize;
			this.HandleVisibility = canvas.HandleVisibility;
			this.MaximumSize = canvas.MaximumSize;
			this.MinimumSize = canvas.MinimumSize;
			this.Movable = canvas.Movable;
			this.Rectangle = new Rectangle(new Point(0,0), this.MinimumSize);
			this.Resizable = canvas.Resizable;
			this.TextLocked = canvas.TextLocked;
			this.TextPlacement = canvas.TextPlacement;
			this.TextVisible = canvas.TextVisible;
		}
		internal Cursor GetCursor(HitTestCode ht)
		{
				switch(ht)
				{
					default:
						return Cursors.Default;
					case HitTestCode.Move:
						return Cursors.SizeAll;
					case HitTestCode.SizeNW:
						return Cursors.SizeNWSE;
					case HitTestCode.SizeN:
						return Cursors.SizeNS;
					case HitTestCode.SizeNE:
						return Cursors.SizeNESW;
					case HitTestCode.SizeE:
						return Cursors.SizeWE;
					case HitTestCode.SizeSE:
						return Cursors.SizeNWSE;
					case HitTestCode.SizeS:
						return Cursors.SizeNS;
					case HitTestCode.SizeSW:
						return Cursors.SizeNESW;
					case HitTestCode.SizeW:
						return Cursors.SizeWE;
				}
		}
		static Shape()
		{
			DefaultStringFormat.Alignment = StringAlignment.Center;
			DefaultStringFormat.FormatFlags = StringFormatFlags.FitBlackBox | StringFormatFlags.NoWrap;
			DefaultStringFormat.HotkeyPrefix = System.Drawing.Text.HotkeyPrefix.None;
			DefaultStringFormat.LineAlignment = StringAlignment.Center;
			DefaultStringFormat.Trimming = StringTrimming.EllipsisCharacter;
		}
		private static Rectangle HandleAround(Point p, Size s)
		{
			return new Rectangle(p.X - s.Width / 2, p.Y - s.Height / 2, s.Width, s.Height);
		}
		private static Rectangle[] HandlesAround(Rectangle r, Size s)
		{
			int midX = r.X + r.Width / 2, midY = r.Y + r.Height / 2, maxX = r.X + r.Width, maxY = r.Y + r.Height;
			return new Rectangle[8]{
									   HandleAround(new Point(r.X, r.Y), s),
									   HandleAround(new Point(midX, r.Y), s),
									   HandleAround(new Point(maxX, r.Y), s),
									   HandleAround(new Point(maxX, midY), s),
									   HandleAround(new Point(maxX, maxY), s),
									   HandleAround(new Point(midX, maxY), s),
									   HandleAround(new Point(r.X, maxY), s),
									   HandleAround(new Point(r.X, midY), s)};
		}
		private Rectangle dragR;
		private Point p, q, dragP;
		private HitTestCode dragHt;
		internal bool Drag(Point anchor)
		{
			bool update = false;
			bool moved = false;
			bool resized = false;
			switch(this.dragHt)
			{
				case HitTestCode.Move:
					update = true;
					this.p = new Point(this.dragR.X + anchor.X - dragP.X, this.dragR.Y + anchor.Y - dragP.Y);
					this.q = new Point(this.p.X + this.dragR.Width, this.p.Y + this.dragR.Height);
					moved = true;
					break;
				case HitTestCode.SizeSE:
					anchor.Offset(-this.borderThickness, -this.borderThickness);
					anchor.X = Math.Max(anchor.X, this.p.X + this.minimumSize.Width);
					anchor.Y = Math.Max(anchor.Y, this.p.Y + this.minimumSize.Height);
					if(anchor != this.q)
					{
						this.q = anchor;
						update = true;
						resized = true;
					}
					break;
				case HitTestCode.SizeNW:
					anchor.Offset(this.borderThickness, this.borderThickness);
					anchor.X = Math.Min(anchor.X, this.p.X - this.minimumSize.Width);
					anchor.Y = Math.Min(anchor.Y, this.p.Y - this.minimumSize.Height);
					if(anchor != this.q)
					{
						this.q = anchor;
						update = true;
						moved = true;
						resized = true;
					}
					break;
				case HitTestCode.SizeNE:
					anchor.Offset(-this.borderThickness, this.borderThickness);
					anchor.X = Math.Max(anchor.X, this.p.X + this.minimumSize.Width);
					anchor.Y = Math.Min(anchor.Y, this.p.Y - this.minimumSize.Height);
					if(anchor != this.q)
					{
						this.q = anchor;
						update = true;
						moved = true;
						resized = true;
					}
					break;
				case HitTestCode.SizeSW:
					anchor.Offset(this.borderThickness, -this.borderThickness);
					anchor.X = Math.Min(anchor.X, this.p.X - this.minimumSize.Width);
					anchor.Y = Math.Max(anchor.Y, this.p.Y + this.minimumSize.Height);
					if(anchor != this.q)
					{
						this.q = anchor;
						update = true;
						moved = true;
						resized = true;
					}
					break;
				case HitTestCode.SizeN:
					anchor = new Point(this.dragR.Left, Math.Min(this.dragR.Bottom - this.minimumSize.Height, anchor.Y + this.borderThickness));
					if(anchor != this.q)
					{
						this.q = anchor;
						update = true;
						moved = true;
						resized = true;
					}
					break;
				case HitTestCode.SizeE:
					anchor = new Point(Math.Max(this.p.X + this.minimumSize.Height, anchor.X - this.borderThickness), this.dragR.Bottom);
					if(anchor != this.q)
					{
						this.q = anchor;
						update = true;
						resized = true;
					}
					break;
				case HitTestCode.SizeS:
					anchor = new Point(this.dragR.Right, Math.Max(this.p.Y + this.minimumSize.Height, anchor.Y - this.borderThickness));
					if(anchor != this.q)
					{
						this.q = anchor;
						update = true;
						resized = true;
					}
					break;
				case HitTestCode.SizeW:
					anchor = new Point(Math.Min(this.p.X - this.minimumSize.Width, anchor.X + this.borderThickness), this.dragR.Bottom);
					if(anchor != this.q)
					{
						this.q = anchor;
						update = true;
						resized = true;
						moved = true;
					}
					break;
			}
			if(update)
			{
				this.rectangle = new Rectangle(Math.Max(0, Math.Min(p.X, q.X)), Math.Max(0, Math.Min(p.Y, q.Y)), Math.Min(this.maximumSize.Width, Math.Max(this.minimumSize.Width, Math.Abs(p.X - q.X))), Math.Min(this.maximumSize.Height, Math.Max(this.minimumSize.Height, Math.Abs(p.Y - q.Y))));
			}
			if((null != this.canvas) && (moved || resized))
			{
				this.canvas.NotifyShapeBoundsChanged(this.eventArgs);
			}
			return update;
		}
		public void EndDrag()
		{
			this.dragHt = HitTestCode.None;
		}
		public HitTestCode Dragging
		{
			get
			{
				return this.dragHt;
			}
		}
		public void BeginDrag(Point pt, HitTestCode ht)
		{
			this.dragHt = ht;
			this.dragP = pt;
			this.dragR = this.rectangle;
			switch(ht)
			{
				default:
					this.p = this.rectangle.Location;
					break;
				case HitTestCode.SizeNW:
					this.p = new Point(this.rectangle.Right, this.rectangle.Bottom);
					break;
				case HitTestCode.SizeNE:
					this.p = new Point(this.rectangle.Left, this.rectangle.Bottom);
					break;
				case HitTestCode.SizeSW:
					this.p = new Point(this.rectangle.Right, this.rectangle.Top);
					break;
				case HitTestCode.SizeN:
					this.p = new Point(this.rectangle.Right, this.rectangle.Bottom);
					break;
				case HitTestCode.SizeE:
					goto default;
				case HitTestCode.SizeS:
					goto default;
				case HitTestCode.SizeW:
					this.p = new Point(this.rectangle.Right, this.rectangle.Top);
					break;
			}
		}
		internal bool HandlesVisible
		{
			get
			{
				return (this.resizable && ((HandleVisibility.Always == this.handleVisibility) || (this.selected && (HandleVisibility.WhenSelected == this.handleVisibility))));
			}
		}
		internal HitTestCode HitTest(Point x)
		{
			if(!this.visible)
			{
				return HitTestCode.None;
			}
			if(!this.enabled)
			{
				return HitTestCode.None;
			}
			int maxX = this.rectangle.Left + this.rectangle.Width, maxY = this.rectangle.Top + this.rectangle.Height;
			if(this.HandlesVisible)
			{
				Rectangle[] handles = HandlesAround(this.BorderStrokeRectangle, this.handleSize);
				for(int i = 0; i < handles.Length; i++)
				{
					if(handles[i].Contains(x))
					{
						return (HitTestCode) (i + 1);
					}
				}
			}
			if(this.movable)
			{
				Rectangle stroke = this.BorderStrokeRectangle, inside, outside;
				int d = this.borderThickness / 2, D = this.borderThickness - d;
				outside = Rectangle.FromLTRB(stroke.Left - D, stroke.Top - D, stroke.Right + d, stroke.Bottom + d);
				inside = Rectangle.FromLTRB(stroke.Left + d, stroke.Top + d, stroke.Right - D, stroke.Bottom - D);
				if((x.X >= outside.Left) && (x.X < outside.Right) && (x.Y >= outside.Top) && (x.Y < outside.Bottom) && (((x.X >= outside.Left) && (x.X < inside.Left)) || ((x.X >= inside.Right) && (x.X < outside.Right)) || ((x.Y >= inside.Bottom) && (x.Y < outside.Bottom)) || ((x.Y >= outside.Top) && (x.Y < inside.Top))))
				{
					return HitTestCode.Move;
				}
			}
			if(!this.borderSelect && this.rectangle.Contains(x))
			{
				return HitTestCode.Move;
			}
			return HitTestCode.None;
		}
		public Rectangle Bounds
		{
			get
			{
				Rectangle box = this.rectangle;
				box.Inflate(this.borderThickness+(this.handleSize.Width+1)/2+1, this.borderThickness+(this.handleSize.Height+1)/2+1);
				return box;
			}
		}
		private Rectangle BorderStrokeRectangle
		{
			get
			{
				const int o = 1;
				bool h = this.HandlesVisible;
				int dx = Math.Max(this.borderThickness / 2, h ? (this.handleSize.Width / 2) : o);
				int Dx = Math.Max(this.borderThickness - this.borderThickness / 2, h ? (this.handleSize.Width - this.handleSize.Width / 2) : o);
				int dy = Math.Max(this.borderThickness / 2, h ? (this.handleSize.Height / 2) : o);
				int Dy = Math.Max(this.borderThickness - this.borderThickness / 2, h ? (this.handleSize.Height - this.handleSize.Height / 2) : o);
				Rectangle b = Rectangle.FromLTRB(this.rectangle.Left - Dx, this.rectangle.Top - Dy, this.rectangle.Right + dx, this.rectangle.Bottom + dy);
				return b;
			}
		}
		internal void Draw(Graphics g, bool drawEnabled)
		{
			if(!this.visible)
			{
				return;
			}
			if(this.selected)
			{
				using(Pen p = new Pen(new HatchBrush(this.borderStyle, drawEnabled ? this.foreColor : UrlGrabber.ToGray(this.foreColor), Color.Transparent), this.borderThickness))
				{
					g.DrawRectangle(p, this.BorderStrokeRectangle);
				}
				if(this.HandlesVisible)
				{
					using(Brush b = new SolidBrush(drawEnabled ? this.backColor : UrlGrabber.ToGray(this.backColor)))
					{
						using(Pen p = new Pen(drawEnabled ? this.foreColor : UrlGrabber.ToGray(this.foreColor), -1))
						{
							Rectangle[] handles = HandlesAround(this.BorderStrokeRectangle, this.handleSize);
							for(int x = 0; x < handles.Length; x++)
							{
								handles[x].Size = new Size(handles[x].Size.Width - 1, handles[x].Size.Height - 1);
								g.FillRectangle(b, handles[x]);
								g.DrawRectangle(p, handles[x]);
							}
						}
					}
				}
			}
			if(this.fillColor != Color.Transparent)
			{
				using(Brush b = new SolidBrush(drawEnabled ? this.fillColor : UrlGrabber.ToGray(this.fillColor)))
				{
					g.FillRectangle(b, this.rectangle);
				}
			}
			if(null != this.image)
			{
				if(drawEnabled)
				{
					g.DrawImage(this.image, this.rectangle, this.ImageRectangle, GraphicsUnit.Pixel);
				}
				else
				{
					UrlGrabber.DrawImageGrayscale(g, this.image as Bitmap, this.rectangle, this.ImageRectangle, GraphicsUnit.Pixel);
				}
			}
			using(Pen p = new Pen(drawEnabled ? this.foreColor : UrlGrabber.ToGray(this.foreColor), -1))
			{
				g.DrawRectangle(p, new Rectangle(this.rectangle.Left, this.rectangle.Top, this.rectangle.Width - 1, this.rectangle.Height - 1));
			}
			if(this.textVisible)
			{
				SizeF mf = g.MeasureString(this.text, this.font, this.rectangle.Width, DefaultStringFormat);
				Size m = new Size((int) Math.Ceiling(mf.Width), (int) Math.Ceiling(mf.Height));
				Point p = Point.Empty;
				int midX = this.rectangle.Left + ((this.rectangle.Width - m.Width) / 2),
					midY = this.rectangle.Top + ((this.rectangle.Height - m.Height) / 2),
					maxX = this.rectangle.Right - m.Width - 1,
					maxY = this.rectangle.Bottom -m.Height - 1; 
				switch(this.textPlacement)
				{
					case ContentAlignment.TopLeft:
						p = new Point(this.rectangle.Left, this.rectangle.Top);
						break;
					case ContentAlignment.TopCenter:
						p = new Point(midX, this.rectangle.Top);
						break;
					case ContentAlignment.TopRight:
						p = new Point(maxX, this.rectangle.Top);
						break;
					case ContentAlignment.MiddleLeft:
						p = new Point(this.rectangle.Left, midY);
						break;
					case ContentAlignment.MiddleCenter:
						p = new Point(midX, midY);
						break;
					case ContentAlignment.MiddleRight:
						p = new Point(maxX, midY);
						break;
					case ContentAlignment.BottomLeft:
						p = new Point(this.rectangle.Left, maxY);
						break;
					case ContentAlignment.BottomCenter:
						p = new Point(midX, maxY);
						break;
					case ContentAlignment.BottomRight:
						p = new Point(maxX, maxY);
						break;
				}
				Rectangle r = new Rectangle(p, m);
				using(Brush b = new SolidBrush(drawEnabled ? this.foreColor : UrlGrabber.ToGray(this.foreColor)))
				{
					g.FillRectangle(b, r);
				}
				using(Brush b = new SolidBrush(drawEnabled ? this.backColor : UrlGrabber.ToGray(this.backColor)))
				{
					g.DrawString(this.text, this.font, b, r, DefaultStringFormat);
				}
			}
		}
		public static Rectangle RectangleFromPoints(Point p, Point q)
		{
			return new Rectangle(Math.Min(p.X, q.X), Math.Min(p.Y, q.Y), Math.Abs(q.X - p.X), Math.Abs(q.Y - p.Y));
		}
		public static HitTestCode HitTestCodeFromDragPoints(Point p1, Point p2)
		{
			int dx = p2.X - p1.X, dy = p2.Y - p1.Y;
			if((dx >= 0) && (dy >= 0))
			{
				return HitTestCode.SizeSE;
			}
			else if((dx <= 0) && (dy <= 0))
			{
				return HitTestCode.SizeNW;
			}
			else if((dx >= 0) && (dy <= 0))
			{
				return HitTestCode.SizeNE;
			}
			else
			{
				return HitTestCode.SizeSW;
			}
		}
		[System.Runtime.InteropServices.DllImport("user32")]
		private static extern int GetSystemMetrics(int index);
		public static Size DragSize
		{
			get
			{
				return new Size(GetSystemMetrics(68), GetSystemMetrics(69));
			}
		}
	}
}
