using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;

namespace WebTracker
{
	class Area
	{
		public enum AreaType
		{
			Change,
			Copy,
		}
		public enum HitTestCode
		{
			None = 0,
			HandleNW = 1,
			HandleN = 2,
			HandleNE = 3,
			HandleE = 4,
			HandleSE = 5,
			HandleS = 6,
			HandleSW = 7,
			HandleW = 8,
			Border = 9,
			Caption = 10
		}
		const int HANDLE_SIZE_X = 9;
		const int HANDLE_SIZE_Y = 9;
		const int BORDER_THICKNESS = 5;
		const int MIN_LABEL_WIDTH = 15;
		const int LABEL_FONT_HEIGHT = 8;
		const int LABEL_HORIZ_PADDING = 5;
		const int LABEL_VERT_PADDING = 1;
		const int MIN_AREA_WIDTH = 2 * BORDER_THICKNESS + 2 * LABEL_HORIZ_PADDING + MIN_LABEL_WIDTH + 3;
		const int MIN_AREA_HEIGHT = 2 * BORDER_THICKNESS +  2 * LABEL_VERT_PADDING + LABEL_FONT_HEIGHT + 3;
		static int nextId = 0;
		public class DrawingObjects : IDisposable
		{
			public class __I : IDisposable
			{
				public readonly Pen HandleDrawPen;
				public readonly Pen BorderDrawPen;
				public readonly Brush HandleFillBrush;
				public readonly Brush LabelBackBrush;
				public readonly Brush LabelForeBrush;
				public __I(Color foreColor, Color backColor)
				{
					HandleDrawPen = new Pen(foreColor, 1);
					BorderDrawPen = new Pen(new System.Drawing.Drawing2D.HatchBrush(System.Drawing.Drawing2D.HatchStyle.DarkDownwardDiagonal, foreColor, Color.Transparent), Area.BORDER_THICKNESS);
					HandleFillBrush = new SolidBrush(backColor);
					LabelBackBrush = new SolidBrush(foreColor);
					LabelForeBrush = new SolidBrush(backColor);
				}
				public void Dispose()
				{
					HandleDrawPen.Dispose();
					BorderDrawPen.Dispose();
					HandleFillBrush.Dispose();
					LabelForeBrush.Dispose();
					LabelBackBrush.Dispose();
				}
			}
			public static Color GetForeColorForAreaType(AreaType t)
			{
				switch(t)
				{
					default:
						return Color.Black;
					case AreaType.Change:
						return Color.Plum;
					case AreaType.Copy:
						return Color.CadetBlue;
				}
			}
			public static Color GetBackColorForAreaType(AreaType t)
			{
				switch(t)
				{
					default:
						return Color.White;
					case AreaType.Change:
						return Color.Snow;
					case AreaType.Copy:
						return Color.Snow;
				}
			}
			public readonly __I ChangeArea;
			public readonly __I CopyArea;
			public readonly Font Font;
			public readonly StringFormat StringFormat;
			private Image image;
			private Graphics measureG;
			public DrawingObjects()
			{
				this.ChangeArea = new __I(GetForeColorForAreaType(AreaType.Change), GetBackColorForAreaType(AreaType.Change));
				this.CopyArea = new __I(GetForeColorForAreaType(AreaType.Copy), GetBackColorForAreaType(AreaType.Copy));
				this.Font = new Font("Tahoma", Area.LABEL_FONT_HEIGHT);
				this.image = new Bitmap(10, 10, System.Drawing.Imaging.PixelFormat.Format32bppArgb);
				this.measureG = Graphics.FromImage(this.image);
				this.StringFormat = new StringFormat(StringFormat.GenericDefault);
				this.StringFormat.Alignment = StringAlignment.Center;
				this.StringFormat.FormatFlags = StringFormatFlags.FitBlackBox | StringFormatFlags.NoWrap;
				this.StringFormat.HotkeyPrefix = System.Drawing.Text.HotkeyPrefix.None;
				this.StringFormat.LineAlignment = StringAlignment.Center;
				this.StringFormat.Trimming = StringTrimming.EllipsisCharacter;
			}
			public void Dispose()
			{
				this.ChangeArea.Dispose();
				this.CopyArea.Dispose();
				this.measureG.Dispose();
				this.image.Dispose();
				this.StringFormat.Dispose();
			}
			public Size MeasureLabel(string s, Size maxSize)
			{
				SizeF f = this.measureG.MeasureString(s, this.Font, maxSize.Width, this.StringFormat);
				return new Size(Math.Min(Math.Max((int) (f.Width + 1), Area.MIN_LABEL_WIDTH) + 2 * Area.LABEL_HORIZ_PADDING, maxSize.Width), Math.Min((int) (f.Width + 2 * Area.LABEL_VERT_PADDING), maxSize.Height));
			}
		}
		static Rectangle HandleAround(Point p)
		{
			return new Rectangle(p.X - HANDLE_SIZE_X / 2, p.Y - HANDLE_SIZE_Y / 2, HANDLE_SIZE_X, HANDLE_SIZE_Y);
		}

		static Rectangle[] HandlesAround(Rectangle r)
		{
			int midX = r.X + r.Width / 2, midY = r.Y + r.Height / 2, maxX = r.X + r.Width, maxY = r.Y + r.Height;
			return new Rectangle[8]{
									   HandleAround(new Point(r.X, r.Y)),
									   HandleAround(new Point(midX, r.Y)),
									   HandleAround(new Point(maxX, r.Y)),
									   HandleAround(new Point(maxX, midY)),
									   HandleAround(new Point(maxX, maxY)),
									   HandleAround(new Point(midX, maxY)),
									   HandleAround(new Point(r.X, maxY)),
									   HandleAround(new Point(r.X, midY))};
		}

		public static Cursor CursorFromHitTestCode(HitTestCode ht)
		{
			switch(ht)
			{
				default:
					return Cursors.Default;
				case HitTestCode.Border:
					return Cursors.SizeAll;
				case HitTestCode.Caption:
					return Cursors.IBeam;
				case HitTestCode.HandleNW:
					return Cursors.SizeNWSE;
				case HitTestCode.HandleN:
					return Cursors.SizeNS;
				case HitTestCode.HandleNE:
					return Cursors.SizeNESW;
				case HitTestCode.HandleE:
					return Cursors.SizeWE;
				case HitTestCode.HandleSE:
					return Cursors.SizeNWSE;
				case HitTestCode.HandleS:
					return Cursors.SizeNS;
				case HitTestCode.HandleSW:
					return Cursors.SizeNESW;
				case HitTestCode.HandleW:
					return Cursors.SizeWE;

			}
		}
		private Rectangle r, dragR;
		private Point p, q, dragP;
		private AreaType t;
		private string label;
		private Size labelSize;
		private HitTestCode dragHt;

		public Area(Point anchor, AreaType type)
		{
			this.r = new Rectangle(anchor, new Size(MIN_AREA_WIDTH, MIN_AREA_HEIGHT));
			this.t = type;
			this.label = String.Format("{0}", System.Threading.Interlocked.Increment(ref Area.nextId));
			this.BeginDrag(HitTestCode.HandleSE, anchor);
			this.UpdateDrag(anchor);
		}
		public bool UpdateDrag(Point anchor)
		{
			bool update = false;
			switch(this.dragHt)
			{
				case HitTestCode.Border:
					update = true;
					this.p = new Point(this.dragR.X + anchor.X - dragP.X, this.dragR.Y + anchor.Y - dragP.Y);
					this.q = new Point(this.p.X + this.dragR.Width, this.p.Y + this.dragR.Height);
					break;
				case HitTestCode.HandleSE:
					if((anchor.X > (this.p.X + MIN_AREA_WIDTH)) && (anchor.Y > (this.p.Y + MIN_AREA_HEIGHT)))
					{
						this.q = anchor;
						update = true;
					}
					break;
				case HitTestCode.HandleNW:
					if((anchor.X < (this.p.X - MIN_AREA_WIDTH)) && (anchor.Y < (this.p.Y - MIN_AREA_HEIGHT)))
					{
						update = true;
						this.q = anchor;
					}
					break;
				case HitTestCode.HandleNE:
					if((anchor.X > (this.p.X + MIN_AREA_WIDTH)) && (anchor.Y < (this.p.Y - MIN_AREA_HEIGHT)))
					{
						update = true;
						this.q = anchor;
					}
					break;
				case HitTestCode.HandleSW:
					if((anchor.X < (this.p.X - MIN_AREA_WIDTH)) && (anchor.Y > (this.p.Y + MIN_AREA_HEIGHT)))
					{
						update = true;
						this.q = anchor;
					}
					break;
				case HitTestCode.HandleN:
					if(anchor.Y < (this.p.Y - MIN_AREA_HEIGHT))
					{
						// p is SE, q is NW
						update = true;
						this.q = new Point(this.dragR.Left, anchor.Y);
					}
					break;
				case HitTestCode.HandleE:
					if(anchor.X > (this.p.X + MIN_AREA_HEIGHT))
					{
						// p is NW, q is SE
						update = true;
						this.q = new Point(anchor.X, this.dragR.Bottom);
					}
					break;
				case HitTestCode.HandleS:
					if(anchor.Y > (this.p.Y + MIN_AREA_HEIGHT))
					{
						// p is NW, q is SE S->E
						update = true;
						this.q = new Point(this.dragR.Right, anchor.Y);
					}
					break;
				case HitTestCode.HandleW:
					if(anchor.X < (this.p.X - MIN_AREA_WIDTH))
					{
						// p is NE, q is SW
						update = true;
						this.q = new Point(anchor.X, this.dragR.Bottom);
					}
					break;
			}
			if(update)
			{
				this.r = new Rectangle(Math.Min(p.X, q.X), Math.Min(p.Y, q.Y), Math.Max(MIN_AREA_WIDTH, Math.Abs(p.X - q.X)), Math.Max(MIN_AREA_HEIGHT, Math.Abs(p.Y - q.Y)));
				using(DrawingObjects o = new DrawingObjects())
				{
					this.labelSize = o.MeasureLabel(this.label, this.r.Size);
				}
			}
			return update;
		}
		public void EndDrag()
		{
			this.dragHt = HitTestCode.None;
		}
		public void BeginDrag(HitTestCode ht, Point pt)
		{
			this.dragHt = ht;
			this.dragP = pt;
			this.dragR = this.r;
			switch(ht)
			{
				default:
					this.p = this.r.Location;
					break;
				case HitTestCode.HandleNW:
					this.p = new Point(this.r.Right, this.r.Bottom);
					break;
				case HitTestCode.HandleNE:
					this.p = new Point(this.r.Left, this.r.Bottom);
					break;
				case HitTestCode.HandleSW:
					this.p = new Point(this.r.Right, this.r.Top);
					break;
				case HitTestCode.HandleN:
					this.p = new Point(this.r.Right, this.r.Bottom);
					break;
				case HitTestCode.HandleE:
					goto default;
				case HitTestCode.HandleS:
					goto default;
				case HitTestCode.HandleW:
					this.p = new Point(this.r.Right, this.r.Top);
					break;
			}
		}
		public HitTestCode HitTest(Point x)
		{
			Rectangle[] handles = HandlesAround(this.r);
			for(int i = 0; i < handles.Length; i++)
			{
				if(handles[i].Contains(x))
				{
					return (HitTestCode) (i + 1);
				}
			}
			int maxX = r.X + r.Width, maxY = r.Y + r.Height;
			if((x.X >= r.X) && (x.X <= maxX) && (x.Y >= r.Y) && (x.Y <= maxY))
			{
				if(((x.X >= r.X) && (x.X <= r.X + BORDER_THICKNESS)) || ((x.X <= maxX) && (x.X >= maxX - BORDER_THICKNESS)) || ((x.Y >= r.Y) && (x.Y <= r.Y + BORDER_THICKNESS)) || ((x.Y <= maxY) && (x.Y >= maxY - BORDER_THICKNESS)))
				{
					return HitTestCode.Border;
				}
			}
			return HitTestCode.None;
		}
		public void Draw(Graphics g, DrawingObjects o)
		{
			DrawingObjects.__I i = (this.t == AreaType.Change) ? o.ChangeArea : o.CopyArea;
			g.DrawRectangle(i.BorderDrawPen, this.r);
			Rectangle[] handles = HandlesAround(this.r);
			for(int x = 0; x < handles.Length; x++)
			{
				g.FillRectangle(i.HandleFillBrush, handles[x]);
				g.DrawRectangle(i.HandleDrawPen, handles[x]);
			}
			Point[] px = new Point[4];
			px[0] = new Point(this.r.X + BORDER_THICKNESS, this.r.Y + BORDER_THICKNESS);
			px[1] = new Point(px[0].X + this.labelSize.Width, px[0].Y);
			px[2] = new Point(px[1].X - LABEL_HORIZ_PADDING, px[1].Y + this.labelSize.Height);
			px[3] = new Point(px[0].X + LABEL_HORIZ_PADDING, px[2].Y);
			using(GraphicsPath p = new GraphicsPath(px, new byte[]{(byte) PathPointType.Start, (byte) PathPointType.Line, (byte) PathPointType.Line, ((byte) PathPointType.Line) | ((byte) PathPointType.CloseSubpath)}, FillMode.Alternate))
			{
				g.FillPath(i.LabelBackBrush, p);
			}
			g.DrawString(this.label, o.Font, i.LabelForeBrush, new Rectangle(px[0], this.labelSize), o.StringFormat);
		}
		public override bool Equals(object obj)
		{
			Area r2 = obj as Area;
			if(null == r2)
			{
				return false;
			}
			else
			{
				return ((r2.r == this.r) && (r2.label == this.label));
			}
		}
		public override int GetHashCode()
		{
			return this.label.GetHashCode() + this.r.GetHashCode();
		}
		public static bool operator==(Area r1, Area r2)
		{
			if(object.ReferenceEquals(r1, null))
			{
				if(object.ReferenceEquals(r2, null))
				{
					return true;
				}
				else
				{
					return false;
				}
			}
			else if(object.ReferenceEquals(r2, null))
			{
				return false;
			}
			else
			{
				return ((r1.r == r2.r) && (r1.label == r2.label));
			}
		}
		public static bool operator!=(Area r1, Area r2)
		{
			if(object.ReferenceEquals(r1, null))
			{
				if(object.ReferenceEquals(r2, null))
				{
					return false;
				}
				else
				{
					return true;
				}
			}
			else if(object.ReferenceEquals(r2, null))
			{
				return true;
			}
			else
			{
				return ((r1.r != r2.r) || (r1.label == r2.label));
			}
		}
		public string Label
		{
			get
			{
				return this.label;
			}
			set
			{
				this.label = value;
			}
		}
		public Rectangle Rectangle
		{
			get
			{
				return this.r;
			}
		}
		public AreaType Type
		{
			get
			{
				return this.t;
			}
		}
	}
}
