using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.Drawing.Text;
using System.Data;
using System.Windows.Forms;

namespace WebTracker
{
	public class Canvas : System.Windows.Forms.UserControl
	{
		private System.ComponentModel.Container components = null;
		ShapeCollection shapes;

		public Canvas()
		{
			this.SetStyle(ControlStyles.UserPaint | ControlStyles.Selectable | ControlStyles.AllPaintingInWmPaint | ControlStyles.DoubleBuffer | ControlStyles.ResizeRedraw, true);
			this.shapes = new ShapeCollection(this);
			InitializeComponent();
		}

		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Component Designer generated code
		/// <summary> 
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			// 
			// Canvas
			// 
			this.AutoScroll = true;
			this.BackColor = System.Drawing.Color.White;
			this.Cursor = System.Windows.Forms.Cursors.Cross;
			this.ForeColor = System.Drawing.Color.Black;
			this.Name = "Canvas";

		}
		#endregion

		private Size handleSize = new Size(9, 9);
		public Size HandleSize
		{
			get
			{
				return this.handleSize;
			}
			set
			{
				this.handleSize = value;
				this.Refresh();
			}
		}
		private int borderThickness = 5;
		public int BorderThickness
		{
			get
			{
				return this.borderThickness;
			}
			set
			{
				this.borderThickness = value;
				this.Refresh();
			}
		}
		private HatchStyle borderStyle = HatchStyle.DarkDownwardDiagonal;
		public HatchStyle BorderStyle
		{
			get
			{
				return this.borderStyle;
			}
			set
			{
				this.borderStyle = value;
				this.Refresh();
			}
		}
		private bool textVisible = true;
		public bool TextVisible
		{
			get
			{
				return this.textVisible;
			}
			set
			{
				this.textVisible = value;
				this.Refresh();
			}
		}
		private bool textLocked = false;
		public bool TextLocked
		{
			get
			{
				return this.textLocked;
			}
			set
			{
				this.textLocked = value;
			}
		}
		private ContentAlignment textPlacement = ContentAlignment.TopCenter;
		public ContentAlignment TextPlacement
		{
			get
			{
				return this.textPlacement;
			}
			set
			{
				this.textPlacement = value;
				this.Refresh();
			}
		}
		private bool movable = true;
		public bool Movable
		{
			get
			{
				return this.movable;
			}
			set
			{
				this.movable = value;
			}
		}
		private bool resizable = true;
		public bool Resizable
		{
			get
			{
				return this.resizable;
			}
			set
			{
				this.resizable = value;
				this.Refresh();
			}
		}
		private HandleVisibility handleVisibility = HandleVisibility.WhenSelected;
		public HandleVisibility HandleVisibility
		{
			get
			{
				return this.handleVisibility;
			}
			set
			{
				this.handleVisibility = value;
				this.Refresh();
			}
		}
		private bool borderSelect = false;
		public bool BorderSelect
		{
			get
			{
				return this.borderSelect;
			}
			set
			{
				this.borderSelect = value;
			}
		}
		private Size minimumSize = new Size(12, 12);
		public Size MinimumSize
		{
			get
			{
				return this.minimumSize;
			}
			set
			{
				this.minimumSize = value;
			}
		}
		private Size maximumSize = new Size(int.MaxValue, int.MaxValue);
		public Size MaximumSize
		{
			get
			{
				return this.maximumSize;
			}
			set
			{
				this.maximumSize = value;
			}
		}
		internal void NotifyShapeAdded(ShapeEventArgs e)
		{
			this.UpdateCanvasSize();
			this.OnShapeAdded(e);
		}
		internal void NotifyShapeRemoved(ShapeEventArgs e)
		{
			this.UpdateCanvasSize();
			this.OnShapeRemoved(e);
		}
		internal void NotifyShapeBoundsChanged(ShapeEventArgs e)
		{
			this.UpdateCanvasSize();
			this.OnShapeBoundsChanged(e);
		}
		internal void NotifySelectionChanged(ShapeEventArgs e)
		{
			this.OnSelectionChanged(e);
		}
		protected virtual void OnShapeAdded(ShapeEventArgs e)
		{
			if(null != this.ShapeAdded)
			{
				this.ShapeAdded(this, e);
			}
		}
		public event ShapeEventHandler ShapeAdded;
		protected virtual void OnBeginDragShape(ShapeEventArgs e)
		{
			if(null != this.BeginDragShape)
			{
				this.BeginDragShape(this, e);
			}
		}
		public event ShapeEventHandler BeginDragShape;
		protected virtual void OnDragShape(ShapeEventArgs e)
		{
			if(null != this.DragShape)
			{
				this.DragShape(this, e);
			}
		}
		public event ShapeEventHandler DragShape;
		protected virtual void OnEndDragShape(ShapeEventArgs e)
		{
			if(null != this.EndDragShape)
			{
				this.EndDragShape(this, e);
			}
		}
		public event ShapeEventHandler EndDragShape;
		protected virtual void OnShapeBoundsChanged(ShapeEventArgs e)
		{
			if(null != this.ShapeBoundsChanged)
			{
				this.ShapeBoundsChanged(this, e);
			}
		}
		public event ShapeEventHandler ShapeBoundsChanged;
		protected virtual void OnShapeRemoved(ShapeEventArgs e)
		{
			if(null != this.ShapeRemoved)
			{
				this.ShapeRemoved(this, e);
			}
		}
		public event ShapeEventHandler ShapeRemoved;
		protected virtual void OnSelectionChanged(ShapeEventArgs e)
		{
			if(null != this.SelectionChanged)
			{
				this.SelectionChanged(this, e);
			}
		}
		public event ShapeEventHandler SelectionChanged;
		protected virtual void OnBeginDrawing(ShapeEventArgs e)
		{
			if(null != this.BeginDrawing)
			{
				this.BeginDrawing(this, e);
			}
		}
		public event ShapeEventHandler BeginDrawing;

		Point mouseDownPoint = Point.Empty;
		bool dragging = false;

		protected override void OnMouseDown(System.Windows.Forms.MouseEventArgs e)
		{
			if(this.DesignMode)
			{
				return;
			}
			if(!this.Enabled)
			{
				return;
			}
			mouseDownPoint = new Point(e.X - this.AutoScrollPosition.X, e.Y - this.AutoScrollPosition.Y);
			this.BeginUpdate();
			this.Focus();
			this.Select();
			dragging = false;
			this.Capture = true;
			if(!drawMode)
			{
				foreach(Shape s in this.shapes)
				{
					HitTestCode ht = s.HitTest(mouseDownPoint);
					if(ht != HitTestCode.None)
					{
						this.EndUpdate();
						return;
					}
				}
				base.OnMouseDown(e);
			}
			else
			{
				foreach(Shape s in this.shapes)
				{
					if(s.Selected)
					{
						s.Selected = false;
					}
				}
			}
			this.EndUpdate();
		}

		protected override void OnMouseMove(System.Windows.Forms.MouseEventArgs e)
		{
			if(this.DesignMode)
			{
				return;
			}
			if(!this.Enabled)
			{
				return;
			}
			if(!this.CanUpdate)
			{
				return;
			}
			Point currentMousePoint = new Point(e.X - this.AutoScrollPosition.X, e.Y - this.AutoScrollPosition.Y);
			this.BeginUpdate();
			bool selected = false;
			int selcount = 0;
			bool alreadyDragging = false;
			foreach(Shape s in this.shapes)
			{
				if(s.Selected)
				{
					selected = true;
					selcount++;
				}
				if(HitTestCode.None != s.Dragging)
				{
					alreadyDragging = true;
				}
			}
			Rectangle dragRectangle = Shape.RectangleFromPoints(currentMousePoint, mouseDownPoint);
			bool bigEnoughToDrag = (dragRectangle.Width >= Shape.DragSize.Width) || (dragRectangle.Height >= Shape.DragSize.Height);
			if((!dragging && (MouseButtons.None != e.Button) && bigEnoughToDrag))
			{
				dragging = true;
			}
			if(drawMode)
			{
				if(!dragging)
				{
					if(this.Cursor != Cursors.Cross)
					{
						this.Cursor = Cursors.Cross;
					}
					base.OnMouseMove(e);
					this.EndUpdate();
					return;
				}
				if(!selected && (MouseButtons.Left == e.Button))
				{
					HitTestCode ht = Shape.HitTestCodeFromDragPoints(mouseDownPoint, currentMousePoint);
					ShapeEventArgs s = new ShapeEventArgs();
					this.OnBeginDrawing(s);
					if(s.Shape != null)
					{
						selected = true;
						s.Shape.Rectangle = dragRectangle;
						s.Shape.Selected = true;
						this.shapes.Add(s.Shape);
						this.OnSelectionChanged(s.Shape.eventArgs);
						s.Shape.BeginDrag(mouseDownPoint, ht);
						this.Cursor = s.Shape.GetCursor(ht);
						alreadyDragging = true;
						this.OnBeginDragShape(s.Shape.eventArgs);
					}
				}
			}
			bool continueProcessing = true;
			if(MouseButtons.None == e.Button)
			{
				foreach(Shape s in this.shapes)
				{
					HitTestCode ht = s.HitTest(currentMousePoint);
					if(ht != HitTestCode.None)
					{
						if(selcount > 1)
						{
							this.Cursor = Cursors.SizeAll;
						}
						else
						{
							this.Cursor = s.GetCursor(ht);
						}
						continueProcessing = false;
						break;
					}
				}
				if(continueProcessing)
				{
					this.Cursor = Cursors.Arrow;
					base.OnMouseMove(e);
				}
			}
			else if(!alreadyDragging)
			{
				HitTestCode ht = HitTestCode.None;
				Shape hts = null;
				bool wasAlreadySelected = false;
				foreach(Shape s in this.shapes)
				{
					ht = s.HitTest(mouseDownPoint);
					if(ht != HitTestCode.None)
					{
						wasAlreadySelected = s.Selected;
						s.Selected = true;
						hts = s;
						break;
					}
				}
				if(!wasAlreadySelected && (Keys.Shift != Control.ModifierKeys) && (Keys.Control != Control.ModifierKeys))
				{
					foreach(Shape s in this.shapes)
					{
						if(!object.ReferenceEquals(s, hts))
						{
							s.Selected = false;
						}
					}
				}
				if(hts != null)
				{
					if(selcount > 1)
					{
						ht = HitTestCode.Move;
						this.Cursor = Cursors.SizeAll;
					}
					foreach(Shape s in this.shapes)
					{
						if(s.Selected)
						{
							selected = true;
							s.BeginDrag(mouseDownPoint, ht);
							alreadyDragging = true;
							this.OnBeginDragShape(s.eventArgs);
						}
					}
				}
				if(!selected)
				{
					base.OnMouseMove(e);
				}
			}
			if(selected && continueProcessing)
			{
				foreach(Shape s in this.shapes)
				{
					if(s.Selected)
					{
						this.Invalidate(s);
						s.Drag(currentMousePoint);
						this.Invalidate(s);
						this.OnDragShape(s.eventArgs);
					}
				}
			}
			this.EndUpdate();
		}

		public void Invalidate(Shape s)
		{
			if(null != s)
			{
				Rectangle r = s.Bounds;
				r.Offset(this.AutoScrollPosition);
				this.Invalidate(r);
			}
		}

		protected override void OnLostFocus(EventArgs e)
		{
			//			this.UnselectAll();
			base.OnLostFocus(e);
		}

		public void SelectAll()
		{
			this.BeginUpdate();
			foreach(Shape s in this.shapes)
			{
				s.Selected = true;
			}
			this.EndUpdate();
		}

		public void ScrollIntoView(Shape s)
		{
			Rectangle r = new Rectangle(-this.AutoScrollPosition.X, -this.AutoScrollPosition.Y, this.ClientSize.Width - 20, this.ClientSize.Height - 20);
			if(!r.Contains(s.Rectangle.Location))
			{
				Point p = new Point(s.Rectangle.Left - this.ClientSize.Width / 2, s.Rectangle.Top - this.ClientSize.Height / 2);
				this.AutoScrollPosition = p;
			}
		}

		public void UnselectAll()
		{
			this.BeginUpdate();
			foreach(Shape s in this.shapes)
			{
				s.Selected = false;
			}
			this.EndUpdate();
		}
		private Image image;
		public Image Image
		{
			get
			{
				return this.image;
			}
			set
			{
				if(null != this.image)
				{
					this.Invalidate(new Rectangle(Point.Empty, this.image.Size));
				}
				this.image = value;
				if(null != this.image)
				{
					this.Invalidate(new Rectangle(Point.Empty, this.image.Size));
				}
				if(this.CanUpdate)
				{
					this.UpdateCanvasSize();
					this.Update();
				}
			}
		}

		protected override void OnMouseUp(System.Windows.Forms.MouseEventArgs e)
		{
			if(this.DesignMode)
			{
				return;
			}
			if(!this.Enabled)
			{
				return;
			}
			Point currentMousePoint = new Point(e.X - this.AutoScrollPosition.X, e.Y - this.AutoScrollPosition.Y);
			this.BeginUpdate();
			this.Capture = false;
			dragging = false;
			this.Cursor = drawMode ? Cursors.Cross : Cursors.Arrow;
			bool alreadyDragging = false;
			foreach(Shape s in this.shapes)
			{
				if(s.Dragging != HitTestCode.None)
				{
					alreadyDragging = true;
				}
			}
			if(alreadyDragging)
			{
				this.DrawMode = false;
				foreach(Shape s in this.shapes)
				{
					if(s.Dragging != HitTestCode.None)
					{
						this.Invalidate(s);
						s.Drag(currentMousePoint);
						s.EndDrag();
						this.Invalidate(s);
						this.OnEndDragShape(s.eventArgs);
					}
				}
			}
			else
			{
				bool deselect = true;
				foreach(Shape s in this.shapes)
				{
					HitTestCode ht = s.HitTest(mouseDownPoint);
					if(ht != HitTestCode.None)
					{
						if(e.Button == MouseButtons.Left)
						{
							if((Control.ModifierKeys != Keys.Shift) && (Control.ModifierKeys != Keys.Control))
							{
								foreach(Shape s2 in this.shapes)
								{
									if(!object.ReferenceEquals(s2, s))
									{
										s2.Selected = false;
									}
								}
							}
							s.Selected = true;
						}
						else if(e.Button == MouseButtons.Right)
						{
							s.Selected = true;
							if(this.ContextMenu != null)
							{
								this.ContextMenu.Show(this, currentMousePoint);
							}
						}
						deselect = false;
						break;
					}
				}
				if(deselect)
				{
					foreach(Shape s2 in this.shapes)
					{
						s2.Selected = false;
					}
					base.OnMouseUp(e);
				}
			}
			this.EndUpdate();
		}

		public bool CanUpdate
		{
			get
			{
				return (0 == System.Threading.Interlocked.CompareExchange(ref this.suspendPaint, 0, 0));
			}
		}

		protected override void OnPaint(System.Windows.Forms.PaintEventArgs e)
		{
			if(this.CanUpdate)
			{
				lock(this.shapes.SyncRoot)
				{
					try
					{
						Rectangle src = e.ClipRectangle;
						src.Offset(-this.AutoScrollPosition.X, -this.AutoScrollPosition.Y);
						if(null != this.image)
						{
							const int w = 4;
							using(Pen p = new Pen(Color.DarkGray, w))
							{
								e.Graphics.DrawLines(p, new Point[]{new Point(this.image.Width + w, 0), new Point(this.image.Width + w, this.image.Height + w), new Point(w, this.image.Height + w)});
							}
						}
						if(this.Enabled)
						{
							using(Brush b = new SolidBrush(this.BackColor))
							{
								e.Graphics.FillRectangle(b, e.ClipRectangle);
							}
							if(null != this.image)
							{
								e.Graphics.DrawImage(this.image, e.ClipRectangle, src, GraphicsUnit.Pixel);
							}
						}
						else
						{
							using(Brush b = new SolidBrush(UrlGrabber.ToGray(this.BackColor)))
							{
								e.Graphics.FillRectangle(b, e.ClipRectangle);
							}
							if(null != this.image)
							{
								UrlGrabber.DrawImageGrayscale(e.Graphics, this.image as Bitmap, e.ClipRectangle, src, GraphicsUnit.Pixel);
							}
						}
					}
					catch(Exception ex)
					{
						System.Diagnostics.Debug.WriteLine(string.Format("WARNING: {0}", ex));
					}
					e.Graphics.InterpolationMode = InterpolationMode.High;
					e.Graphics.TextRenderingHint = TextRenderingHint.ClearTypeGridFit;
					e.Graphics.TranslateTransform(this.AutoScrollPosition.X, this.AutoScrollPosition.Y);
					for(int i = this.shapes.Count - 1; i >= 0; i--)
					{
						try
						{
							this.shapes[i].Draw(e.Graphics, this.shapes[i].Enabled && this.Enabled);
						}
						catch(Exception x)
						{
							System.Diagnostics.Debug.WriteLine(string.Format("WARNING: {0}", x));
						}
					}
				}
			}
			base.OnPaint(e);
		}

		public ShapeCollection Shapes
		{
			get
			{
				return this.shapes;
			}
		}
		private bool drawMode = true;
		public bool DrawMode
		{
			get
			{
				return this.drawMode;
			}
			set
			{
				this.drawMode = value;
				if(this.drawMode)
				{
					this.Cursor = Cursors.Cross;
				}
				else
				{
					this.Cursor = Cursors.Arrow;
				}
			}
		}
		protected override bool IsInputKey(Keys keyData)
		{
			switch(keyData)
			{
				case Keys.Up:
					return true;
				case Keys.Down:
					return true;
				case Keys.Left:
					return true;
				case Keys.Right:
					return true;
				default:
					return base.IsInputKey(keyData);
			}
		}
		private Size canvasSize = Size.Empty;
		[DefaultValue("0, 0")]
		public Size CanvasSize
		{
			get
			{
				if(this.DesignMode)
				{
					return this.canvasSize;
				}
				if(this.canvasSize == Size.Empty)
				{
					if(null != this.image)
					{
						try
						{
							return this.image.Size;
						}
						catch(Exception)
						{
							System.Diagnostics.Debug.Assert(false);
							return Size.Empty;
						}
					}
					else if(0 == this.shapes.Count)
					{
						return Size.Empty;
					}
					else
					{
						Rectangle r = Rectangle.Empty;
						lock(this.shapes.SyncRoot)
						{
							foreach(Shape s in this.shapes)
							{
								r = Rectangle.Union(r, s.Bounds);
							}
						}
						return r.Size;
					}
				}
				else
				{
					return this.canvasSize;
				}
			}
			set
			{
				this.canvasSize = value;
				this.UpdateCanvasSize();
			}
		}

		private int suspendPaint = 0;
		public void BeginUpdate()
		{
			System.Threading.Interlocked.Increment(ref this.suspendPaint);
			System.Threading.Monitor.Enter(this.shapes.SyncRoot);
		}

		public void EndUpdate()
		{
			System.Threading.Monitor.Exit(this.shapes.SyncRoot);
			if(0 == System.Threading.Interlocked.CompareExchange(ref this.suspendPaint, 0, 0))
			{
				this.Refresh();
				this.Update();
			}
			else
			{
				if(0 == System.Threading.Interlocked.Decrement(ref this.suspendPaint))
				{
					this.Refresh();
					this.Update();
				}
			}
		}
		private int alreadyUpdatingCanvasSize = 0;

		private void UpdateCanvasSize()
		{
			if(0 != System.Threading.Interlocked.CompareExchange(ref this.alreadyUpdatingCanvasSize, 1, 0))
			{
				return;
			}
			this.BeginUpdate();
			Size s = this.CanvasSize;
			foreach(Shape q in this.shapes)
			{
				Rectangle r = q.Rectangle;
				int l = r.Left, t = r.Top;
				if(r.Right >= (l + s.Width))
				{
					l = Math.Max(0, s.Width - r.Width - 1);
				}
				if(r.Bottom >= (t + s.Height))
				{
					t = Math.Max(0, s.Height - r.Height - 1);
				}
				if((l != r.Left) || (t != r.Top))
				{
					q.Rectangle = new Rectangle(l, t, r.Width, r.Height);
				}
			}
			this.AutoScrollMinSize = s;
			System.Threading.Interlocked.Exchange(ref this.alreadyUpdatingCanvasSize, 0);
			this.EndUpdate();
		}
	}
}
