using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;
using System.Reflection;
using System.Runtime.InteropServices;
using System.Xml;
using System.Xml.Serialization;

namespace WebTracker
{
	[Serializable]
	public class RegionBase
	{
		public Rectangle Source;
		public string Text;
		public RegionBase()
		{
		}
	}
	[Serializable]
	public class LandmarkRelativeRegionBase : RegionBase
	{
		public string LandmarkName;
		public LandmarkRelativeRegionBase()
		{
		}
		[XmlIgnore] public string SafeLandmarkName
		{
			get
			{
				if((null == LandmarkName) || (LandmarkName.Length == 0))
				{
					return "(none)";
				}
				else
				{
					return LandmarkName;
				}
			}
		}
		public virtual void Update(LandmarkRegion lr)
		{
			if((null != lr) && !lr.Offset.IsEmpty)
			{
				this.Source.Offset(lr.Offset);
			}
		}
	}
	[Serializable]
	public class CopyRegion : LandmarkRelativeRegionBase
	{
		public Rectangle Destination;
		[XmlIgnore] public Shape ThumbnailCanvasShape;
		public CopyRegion()
		{
		}
	}
	[Serializable]
	public class ChangeRegion : LandmarkRelativeRegionBase
	{
		public double Threshold;
		[XmlIgnore] public double PSNR;
		[XmlIgnore] public bool Changed;
		[XmlIgnore] public Image DifferenceImage;
		[XmlIgnore] public Rectangle PrevSource;
		public ChangeRegion()
		{
		}
		public void Update(Collabrary.Photo first, Collabrary.Photo second, LandmarkRegion lr)
		{
			this.PrevSource = this.Source;
			base.Update(lr);
			Collabrary.Photo p = (Collabrary.Photo) first.Copy(this.PrevSource.X, this.PrevSource.Y, this.PrevSource.Width, this.PrevSource.Height, Collabrary.PhotoCopyStyle.Pixels);
			try
			{
				Collabrary.Photo q = (Collabrary.Photo) second.Copy(this.Source.X, this.Source.Y, this.Source.Width, this.Source.Height, Collabrary.PhotoCopyStyle.Pixels);
				try
				{
					q.Resize(p.Width, p.Height, true);
					q.Subtract(p, Missing.Value, Missing.Value);
					this.PSNR = q.PSNR;
					if(double.IsInfinity(this.PSNR))
					{
						this.Changed = false;
					}
					else
					{
						this.Changed = (this.PSNR < this.Threshold);
					}
					if(null != this.DifferenceImage)
					{
						this.DifferenceImage.Dispose();
						this.DifferenceImage = null;
					}
					this.DifferenceImage = Image.FromHbitmap(q.Hbitmap);
				}
				finally
				{
					Marshal.ReleaseComObject(q);
				}
			}
			finally
			{
				Marshal.ReleaseComObject(p);
			}
		}
	}
	[Serializable]
	public class LandmarkRegion : RegionBase
	{
		[XmlIgnore] public Point Offset;
		public LandmarkRegion()
		{
		}
		public void Update(Collabrary.Photo first, Collabrary.Photo second)
		{
			int X = this.Source.Left;
			int Y = this.Source.Top;
			Collabrary.Photo p = (Collabrary.Photo) first.Copy(this.Source.Left, this.Source.Top, this.Source.Width, this.Source.Height, Collabrary.PhotoCopyStyle.Pixels);
			second.Search(p, ref X, ref Y, 100, 100);
			Marshal.ReleaseComObject(p);
			this.Offset = new Point(X - this.Source.Left, Y - this.Source.Top);
			this.Source.Offset(this.Offset);
		}
	}
	[Serializable]
	public class WebPageNotification
	{
		public string Url;
		public string Title;
		public int UpdateInterval;
		public CopyRegion[] CopyRegions;
		public ChangeRegion[] ChangeRegions;
		public LandmarkRegion[] LandmarkRegions;
		public string PngImageData;
		public int NextRegionId;
		public bool ShowFullPageThumbnail;
		[XmlIgnore] private static XmlSerializer Serializer;
		[XmlIgnore] public bool Touched;
		[XmlIgnore] public Image Image;
		[XmlIgnore] public Image PreviousImage;

		public RegionBase NewRegion(Type regionType)
		{
			RegionBase region = regionType.GetConstructor(Type.EmptyTypes).Invoke(null) as RegionBase;
			region.Text = string.Format("region{0}", ++this.NextRegionId);
			this.Add(region);
			return region;
		}
		public static WebPageNotification Open(string fileName)
		{
			if(null == Serializer)
			{
				Serializer = new XmlSerializer(typeof(WebPageNotification));
			}
			FileStream s = new FileStream(fileName, FileMode.Open, FileAccess.Read, FileShare.Read);
			try
			{
				XmlTextReader r = new XmlTextReader(s);
				WebPageNotification n = (WebPageNotification) Serializer.Deserialize(r);
				n.Image = DecodeImage(n.PngImageData);
				n.PngImageData = string.Empty;
				return n;
			}
			finally
			{
				s.Close();
			}
		}
		public void Log()
		{
			System.Text.StringBuilder b = new System.Text.StringBuilder();
			b.Append(string.Format("Time: {0:s}\r\n", DateTime.Now));
			b.Append(string.Format("Url: {0}\r\n", this.Url));
			for(int i = 0; i < this.LandmarkRegions.Length; i++)
			{
				if(this.LandmarkRegions[i].Offset.IsEmpty)
				{
					b.Append(string.Format(" Landmark: {0} at {1} did not move\r\n", this.LandmarkRegions[i].Text, this.LandmarkRegions[i].Source));
				}
				else
				{
					b.Append(string.Format(" Landmark: {0} moved {1} px to {2}\r\n", this.LandmarkRegions[i].Text, this.LandmarkRegions[i].Offset, this.LandmarkRegions[i].Source));
				}
			}
			for(int i = 0; i < this.ChangeRegions.Length; i++)
			{
				if(this.ChangeRegions[i].Changed)
				{
					b.Append(string.Format(" ChangeRegion: {0} at {1} changed (PSNR = {2})\r\n", this.ChangeRegions[i].Text, this.ChangeRegions[i].Source, this.ChangeRegions[i].PSNR));
				}
				else
				{
					b.Append(string.Format(" ChangeRegion: {0} at {1} did not change (PSNR = {2})\r\n", this.ChangeRegions[i].Text, this.ChangeRegions[i].Source, this.ChangeRegions[i].PSNR));
				}
			}
			string logpath = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.Personal), "webtracker.log");
			StreamWriter w = new StreamWriter(logpath, true, System.Text.Encoding.UTF8);
			w.WriteLine(b.ToString());
			w.Close();
		}
		public void SaveAs(string fileName)
		{
			if(null == Serializer)
			{
				Serializer = new XmlSerializer(typeof(WebPageNotification));
			}
			FileStream fs = new FileStream(fileName, FileMode.Create, FileAccess.ReadWrite, FileShare.Read);
			try
			{
				XmlWriter w = new XmlTextWriter(fs, System.Text.Encoding.UTF8);
				this.PngImageData = EncodeImage(this.Image);
				Serializer.Serialize(w, this);
				this.PngImageData = string.Empty;
			}
			finally
			{
				fs.Close();
			}
		}
		public WebPageNotification()
		{
			this.CopyRegions = new CopyRegion[0];
			this.ChangeRegions = new ChangeRegion[0];
			this.LandmarkRegions = new LandmarkRegion[0];
		}
		public void Add(RegionBase region)
		{
			if(region is CopyRegion)
			{
				CopyRegion[] r = new CopyRegion[this.CopyRegions.Length + 1];
				r[0] = region as CopyRegion;
				for(int i = 0; i < this.CopyRegions.Length; i++)
				{
					r[i + 1] = this.CopyRegions[i];
				}
				this.CopyRegions = r;
			}
			else if(region is ChangeRegion)
			{
				ChangeRegion[] r = new ChangeRegion[this.ChangeRegions.Length + 1];
				r[0] = region as ChangeRegion;
				for(int i = 0; i < this.ChangeRegions.Length; i++)
				{
					r[i + 1] = this.ChangeRegions[i];
				}
				this.ChangeRegions = r;
			}
			else if(region is LandmarkRegion)
			{
				LandmarkRegion[] r = new LandmarkRegion[this.LandmarkRegions.Length + 1];
				r[0] = region as LandmarkRegion;
				for(int i = 0; i < this.LandmarkRegions.Length; i++)
				{
					r[i + 1] = this.LandmarkRegions[i];
				}
				this.LandmarkRegions = r;
			}
		}
		public void Remove(RegionBase region)
		{
			if(region is CopyRegion)
			{
				for(int i = 0; i < this.CopyRegions.Length; i++)
				{
					if(region == this.CopyRegions[i])
					{
						CopyRegion[] r = new CopyRegion[this.CopyRegions.Length - 1];
						for(int j = 0; j < i; j++)
						{
							r[j] = this.CopyRegions[j];
						}
						for(int j = i + 1; j < this.CopyRegions.Length; j++)
						{
							r[j - 1] = this.CopyRegions[j];
						}
						this.CopyRegions = r;
					}
				}
			}
			else if(region is ChangeRegion)
			{
				for(int i = 0; i < this.ChangeRegions.Length; i++)
				{
					if(region == this.ChangeRegions[i])
					{
						ChangeRegion[] r = new ChangeRegion[this.ChangeRegions.Length - 1];
						for(int j = 0; j < i; j++)
						{
							r[j] = this.ChangeRegions[j];
						}
						for(int j = i + 1; j < this.ChangeRegions.Length; j++)
						{
							r[j - 1] = this.ChangeRegions[j];
						}
						this.ChangeRegions = r;
					}
				}
			}
			else if(region is LandmarkRegion)
			{
				for(int i = 0; i < this.LandmarkRegions.Length; i++)
				{
					if(region == this.LandmarkRegions[i])
					{
						LandmarkRegion[] r = new LandmarkRegion[this.LandmarkRegions.Length - 1];
						for(int j = 0; j < i; j++)
						{
							r[j] = this.LandmarkRegions[j];
						}
						for(int j = i + 1; j < this.LandmarkRegions.Length; j++)
						{
							r[j - 1] = this.LandmarkRegions[j];
						}
						this.LandmarkRegions = r;
					}
				}
			}
		}
		public void BringToFront(RegionBase region)
		{
			Array a;
			if(region is ChangeRegion)
			{
				a = this.ChangeRegions;
			}
			else if(region is CopyRegion)
			{
				a = this.CopyRegions;
			}
			else if(region is LandmarkRegion)
			{
				a = this.LandmarkRegions;
			}
			else
			{
				return;
			}
			for(int i = 0; i < a.Length; i++)
			{
				if(object.ReferenceEquals(a.GetValue(i), region))
				{
					if(i > 0)
					{
						for(int j = i; j > 0; j--)
						{
							a.SetValue(a.GetValue(j - 1), j);
						}
					}
					a.SetValue(region, 0);
					break;
				}
			}
		}
		public void SendToBack(RegionBase region)
		{
			Array a;
			if(region is ChangeRegion)
			{
				a = this.ChangeRegions;
			}
			else if(region is CopyRegion)
			{
				a = this.CopyRegions;
			}
			else if(region is LandmarkRegion)
			{
				a = this.LandmarkRegions;
			}
			else
			{
				return;
			}
			for(int i = 0; i < a.Length; i++)
			{
				if(object.ReferenceEquals(a.GetValue(i), region))
				{
					if(i < (a.Length - 1))
					{
						for(int j = i; j < (a.Length - 1); j++)
						{
							a.SetValue(a.GetValue(j + 1), j);
						}
					}
					a.SetValue(region, a.Length - 1);
					break;
				}
			}
		}
		public static string EncodeImage(Image i)
		{
			MemoryStream ms = new MemoryStream();
			i.Save(ms, System.Drawing.Imaging.ImageFormat.Png);
			byte[] b = ms.GetBuffer();
			return Convert.ToBase64String(b, 0, b.Length);
		}
		public static string EncodePhoto(Collabrary.Photo p)
		{
			Collabrary.PNG png = new Collabrary.PNGClass();
			Collabrary.Buffer b = png.Compress(p) as Collabrary.Buffer;
			string s = b.Copy(Collabrary.BufferCopyPasteStyle.Base64, 0, 0) as string;
			Marshal.ReleaseComObject(b);
			Marshal.ReleaseComObject(png);
			return s;
		}
		public static Image DecodeImage(string base64)
		{
			byte[] b = Convert.FromBase64String(base64);
			MemoryStream ms = new MemoryStream(b, 0, b.Length, false, false);
			return Image.FromStream(ms);
		}
		public static Collabrary.Photo DecodePhoto(string base64)
		{
			if(string.Empty != base64)
			{
				Collabrary.Buffer b = new Collabrary.BufferClass();
				Collabrary.PNG png = new Collabrary.PNGClass();
				b.Paste(base64, Collabrary.BufferCopyPasteStyle.Base64, 0);
				Collabrary.Photo p = png.Decompress(b) as Collabrary.Photo;
				Marshal.ReleaseComObject(b);
				Marshal.ReleaseComObject(png);
				return p;
			}
			else
			{
				return null;
			}
		}
		[DllImport("kernel32", EntryPoint="RtlMoveMemory")]
		private static extern void MoveMemory(IntPtr dest, IntPtr src, int len);
		public static Collabrary.Photo GetPhoto(Image image, Rectangle r)
		{
			Bitmap b = image as Bitmap;
			Point q = new Point(Math.Max(0, Math.Min(b.Width - 2, r.Left)), Math.Max(0, Math.Min(b.Height - 2, r.Top)));
			r = Rectangle.FromLTRB(q.X, q.Y, Math.Max(q.X + 1, Math.Min(b.Width - 1, r.Right)), Math.Max(q.Y + 1, Math.Min(b.Height - 1, r.Bottom)));
			Collabrary.Photo p = new Collabrary.PhotoClass();
			BitmapData bd = b.LockBits(r, ImageLockMode.ReadOnly, PixelFormat.Format24bppRgb);
			try
			{
				p.SetSize(bd.Width, bd.Height);
				IntPtr ptr;
				int len;
				p.LockMemory(out ptr, out len);
				try
				{
					int pstride = ((p.Width * 3 + 3) >> 2) << 2;
					for(int i = 0; i < bd.Height; i++)
					{
						MoveMemory((IntPtr) (((int) ptr) + ((p.Height - 1 - i) * pstride)), (IntPtr) (((int) bd.Scan0) + (i * bd.Stride)), pstride);
					}
				}
				finally
				{
					p.UnlockMemory();
				}
			}
			finally
			{
				b.UnlockBits(bd);
			}
			return p;
		}
		[XmlIgnore] public Image CompositeImage
		{
			get
			{
				if(this.CopyRegions.Length == 0)
				{
					return null;
				}
				int nx = int.MaxValue, ny = int.MaxValue, xx = int.MinValue, xy = int.MinValue;
				for(int i = 0; i < this.CopyRegions.Length; i++)
				{
					Rectangle r = this.CopyRegions[i].Destination; 
					nx = Math.Min(nx, r.Left);
					ny = Math.Min(ny, r.Top);
					xx = Math.Max(xx, r.Right);
					xy = Math.Max(xy, r.Bottom);
				}
				Size s = new Size(2 + xx - nx, 2 + xy - ny);
				Point ofs = new Point(-nx, -ny);
				Bitmap bm = new Bitmap(s.Width, s.Height, PixelFormat.Format24bppRgb);
				using(Graphics g = Graphics.FromImage(bm))
				{
					g.InterpolationMode = System.Drawing.Drawing2D.InterpolationMode.High;
					g.Clear(Color.Black);
					for(int i = this.CopyRegions.Length - 1; i >= 0; i--)
					{
						Rectangle d = this.CopyRegions[i].Destination;
						d.Offset(ofs);
						g.DrawImage(this.Image, d, this.CopyRegions[i].Source, GraphicsUnit.Pixel);
					}
				}
				return bm;
			}
		}
		public LandmarkRegion GetLandmark(string name)
		{
			for(int i = 0; i < this.LandmarkRegions.Length; i++)
			{
				if(name == this.LandmarkRegions[i].Text)
				{
					return this.LandmarkRegions[i];
				}
			}
			return null;
		}
		[XmlIgnore] public Image FullPageImage
		{
			get
			{
				return this.Image;
			}
			set
			{
				if(null != this.PreviousImage)
				{
					this.PreviousImage.Dispose();
				}
				this.PreviousImage = this.Image;
				this.Image = value;
				if(null != this.Image)
				{
					Collabrary.Photo second = GetPhoto(this.Image, new Rectangle(Point.Empty, this.Image.Size));
					if(null != this.PreviousImage)
					{
						Collabrary.Photo first = GetPhoto(this.PreviousImage, new Rectangle(Point.Empty, this.PreviousImage.Size));
						for(int i = 0; i < this.LandmarkRegions.Length; i++)
						{
							this.LandmarkRegions[i].Update(first, second);
							this.Touched = this.Touched || !this.LandmarkRegions[i].Offset.IsEmpty;
						}
						for(int i = 0; i < this.ChangeRegions.Length; i++)
						{
							this.ChangeRegions[i].Update(first, second, this.GetLandmark(this.ChangeRegions[i].LandmarkName));
						}
						for(int i = 0; i < this.CopyRegions.Length; i++)
						{
							this.CopyRegions[i].Update(this.GetLandmark(this.CopyRegions[i].LandmarkName));
						}
						Marshal.ReleaseComObject(first);
					}
					Marshal.ReleaseComObject(second);
				}
			}
		}
		[XmlIgnore] public Image FullPageThumbnail
		{
			get
			{
				const double LETTER_RATIO = 1.29411765;
				const double THUMBNAIL_RATIO = LETTER_RATIO;
				const int THUMBNAIL_WIDTH = 400;
				int srcHeight = Math.Min(this.Image.Height, (int) Math.Ceiling(this.Image.Width * THUMBNAIL_RATIO));
				int dstHeight = (int) Math.Ceiling(((double) (srcHeight * THUMBNAIL_WIDTH)) / ((double) this.Image.Width ));
				Bitmap thumbnailImage = new Bitmap(THUMBNAIL_WIDTH, dstHeight, System.Drawing.Imaging.PixelFormat.Format24bppRgb);
				using(Graphics g = Graphics.FromImage(thumbnailImage))
				{
					g.InterpolationMode = System.Drawing.Drawing2D.InterpolationMode.High;
					g.DrawImage(this.Image, new Rectangle(0, 0, thumbnailImage.Width, thumbnailImage.Height), new Rectangle(0, 0, this.Image.Width, srcHeight), GraphicsUnit.Pixel);
					g.Flush(System.Drawing.Drawing2D.FlushIntention.Sync);
				}
				return thumbnailImage;
			}
		}
		public void Update()
		{
			string title;
			this.FullPageImage = UrlGrabber.Grab(this.Url, out title);
			if((null == this.Title) || (this.Title == string.Empty))
			{
				this.Title = title;
			}
		}
	}
}
